<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Services\CryptoService;
use Illuminate\Http\Request;
use App\Models\CryptoAsset;
use Illuminate\Support\Facades\Auth;

class ReceiveController extends Controller
{
    private $cryptoService;

    public function __construct(CryptoService $cryptoService)
    {
        $this->cryptoService = $cryptoService;
    }

    private function formatAsset($symbol, $name, $coinId, $balance, $address = null, $network = null, $networkSymbol = null)
    {
        $prices = $this->cryptoService->getPrices();
        $price = $prices[$coinId]['usd'] ?? 0;
        $change = $prices[$coinId]['usd_24h_change'] ?? 0;

        $asset = [
            'symbol' => $symbol,
            'name' => $name . ($network ? " ($network)" : ''),
            'balance' => $balance,
            'value' => $balance * $price,
            'price' => $price,
            'change' => $change < 0 ? number_format($change, 2) : '+' . number_format($change, 2),
            'icon_url' => $this->cryptoService->getIconUrl($coinId),
            'network' => $network
        ];

        if ($networkSymbol) {
            $asset['network_url'] = $this->cryptoService->getIconUrl($networkSymbol);
        }

        if ($address) {
            $asset['address'] = $address;
        }

        return $asset;
    }

    private function mapCryptoAssets(CryptoAsset $cryptoAsset, $includeAddresses = false)
    {
        $assets = [];

        // Bitcoin
        if ($cryptoAsset->btc_status) {
            $assets[] = $this->formatAsset(
                'BTC', 'Bitcoin', 'bitcoin', 
                $cryptoAsset->btc_balance,
                $includeAddresses ? $cryptoAsset->btc_address : null
            );
        }

        // USDT with different networks
        $usdtNetworks = [
            'trc20' => ['status' => 'usdt_trc20_status', 'balance' => 'usdt_trc20_balance', 'address' => 'usdt_trc20_address', 'network' => 'TRC20', 'symbol' => 'tron'],
            'bep20' => ['status' => 'usdt_bep20_status', 'balance' => 'usdt_bep20_balance', 'address' => 'usdt_bep20_address', 'network' => 'BNB', 'symbol' => 'binancecoin'],
            'erc20' => ['status' => 'usdt_erc20_status', 'balance' => 'usdt_erc20_balance', 'address' => 'usdt_erc20_address', 'network' => 'ERC20', 'symbol' => 'ethereum']
        ];

        foreach ($usdtNetworks as $network => $config) {
            if ($cryptoAsset->{$config['status']}) {
                $assets[] = $this->formatAsset(
                    'USDT',
                    'Tether',
                    'tether',
                    $cryptoAsset->{$config['balance']},
                    $includeAddresses ? $cryptoAsset->{$config['address']} : null,
                    $config['network'],
                    $config['symbol']
                );
            }
        }

        // Other cryptocurrencies mapping - NOW INCLUDES XDC AND XRP
        $cryptoMapping = [
            'eth' => ['symbol' => 'ETH', 'name' => 'Ethereum', 'id' => 'ethereum'],
            'trx' => ['symbol' => 'TRX', 'name' => 'Tron', 'id' => 'tron'],
            'bnb' => ['symbol' => 'BNB', 'name' => 'Binance Coin', 'id' => 'binancecoin'],
            'dot' => ['symbol' => 'DOT', 'name' => 'Polkadot', 'id' => 'polkadot'],
            'bch' => ['symbol' => 'BCH', 'name' => 'Bitcoin Cash', 'id' => 'bitcoin-cash'],
            'ltc' => ['symbol' => 'LTC', 'name' => 'Litecoin', 'id' => 'litecoin'],
            'xlm' => ['symbol' => 'XLM', 'name' => 'Stellar', 'id' => 'stellar'],
            'dash' => ['symbol' => 'DASH', 'name' => 'Dash', 'id' => 'dash'],
            'sol' => ['symbol' => 'SOL', 'name' => 'Solana', 'id' => 'solana'],
            'doge' => ['symbol' => 'DOGE', 'name' => 'Dogecoin', 'id' => 'dogecoin'],
            'xdc' => ['symbol' => 'XDC', 'name' => 'XDC Network', 'id' => 'xdc-network'],
            'xrp' => ['symbol' => 'XRP', 'name' => 'Ripple', 'id' => 'ripple']
        ];

        foreach ($cryptoMapping as $key => $crypto) {
            $statusField = "{$key}_status";
            $balanceField = "{$key}_balance";
            $addressField = "{$key}_address";
            
            if ($cryptoAsset->$statusField) {
                $assets[] = $this->formatAsset(
                    $crypto['symbol'],
                    $crypto['name'],
                    $crypto['id'],
                    $cryptoAsset->$balanceField,
                    $includeAddresses ? $cryptoAsset->$addressField : null
                );
            }
        }

        return $assets;
    }

    public function payidAssets()
    {
        $user = Auth::user();
        $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();
        
        if (!$cryptoAsset) {
            return redirect()->route('dashboard')->with('error', 'No crypto assets found');
        }

        return view('user.receive.assets-payid', [
            'assets' => $this->mapCryptoAssets($cryptoAsset)
        ]);
    }

    public function externalAssets()
    {
        $user = Auth::user();
        $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();
        
        if (!$cryptoAsset) {
            return redirect()->route('dashboard')->with('error', 'No crypto assets found');
        }

        return view('user.receive.assets-external', [
            'assets' => $this->mapCryptoAssets($cryptoAsset, true)
        ]);
    }

    public function payidDetails($symbol, $network = 'native')
    {
        $user = Auth::user();
        $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();
        
        if (!$cryptoAsset) {
            return redirect()->route('dashboard')->with('error', 'No crypto assets found');
        }
        
        if ($network === 'BNB') {
            $network = 'BEP20';
        }

        $asset = $this->getAssetDetails($cryptoAsset, $symbol, $network);

        if (!$asset) {
            return redirect()->route('dashboard')->with('error', 'Asset not found');
        }

        // Generate QR data
        $qrData = [
            'payId' => $user->uuid,
            'network' => $asset['network'] ?? 'native',
            'symbol' => $asset['symbol'],
            'type' => 'DB_PAYMENT'
        ];

        return view('user.receive.payid-details', [
            'asset' => $asset,
            'user' => $user,
            'qrData' => json_encode($qrData)
        ]);
    }

    public function externalDetails($symbol, $network = 'native')
    {
        $user = Auth::user();
        $cryptoAsset = CryptoAsset::where('user_id', $user->id)->first();
        
        if (!$cryptoAsset) {
            return redirect()->route('dashboard')->with('error', 'No crypto assets found');
        }

        if ($network === 'BNB') {
            $network = 'BEP20';
        }

        $asset = $this->getAssetDetails($cryptoAsset, $symbol, $network, true);

        if (!$asset) {
            return redirect()->route('dashboard')->with('error', 'Asset not found');
        }

        return view('user.receive.external-details', [
            'asset' => $asset
        ]);
    }

    private function getAssetDetails(CryptoAsset $cryptoAsset, $symbol, $network = 'native', $includeAddress = false)
    {
        // Handle USDT with different networks
        if (strtoupper($symbol) === 'USDT' && $network !== 'native') {
            $balanceField = 'usdt_' . strtolower($network) . '_balance';
            $statusField = 'usdt_' . strtolower($network) . '_status';
            $addressField = 'usdt_' . strtolower($network) . '_address';
            
            if ($cryptoAsset->$statusField) {
                $networkSymbol = match(strtolower($network)) {
                    'trc20' => 'tron',
                    'bep20' => 'binancecoin',
                    'erc20' => 'ethereum',
                    default => null
                };

                return $this->formatAsset(
                    'USDT',
                    'Tether',
                    'tether',
                    $cryptoAsset->$balanceField,
                    $includeAddress ? $cryptoAsset->$addressField : null,
                    strtoupper($network),
                    $networkSymbol
                );
            }
        }
        // Handle other assets
        else {
            $balanceField = strtolower($symbol) . '_balance';
            $addressField = strtolower($symbol) . '_address';
            
            if ($cryptoAsset->$balanceField) {
                $coinId = $this->cryptoService->getStandardSymbol($symbol);
                return $this->formatAsset(
                    strtoupper($symbol),
                    ucfirst($symbol),
                    $coinId,
                    $cryptoAsset->$balanceField,
                    $includeAddress ? $cryptoAsset->$addressField : null
                );
            }
        }

        return null;
    }
}